<?php

namespace App\Http\Controllers;

use App\Models\Group;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class GroupController extends Controller
{
    public function index(): JsonResponse
    {
        $groups = Group::with('creator')->get();
        return response()->json([
            'status' => 'success',
            'data' => $groups,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $group = Group::with('creator')->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $group,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Name' => 'required|string|max:255',
            'Description' => 'nullable|string',
            'Created_by' => 'required|exists:Users,User_id',
            'members' => 'sometimes|array',
            'members.*' => 'exists:Users,User_id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $group = Group::create($request->only([
            'Name',
            'Description',
            'Created_by',
        ]));

        if ($request->has('members')) {
            $group->members()->attach($request->input('members'));
        }

        return response()->json([
            'status' => 'success',
            'data' => $group,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $group = Group::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Name' => 'sometimes|string|max:255',
            'Description' => 'nullable|string',
            'Created_by' => 'sometimes|exists:Users,User_id',
            'members' => 'sometimes|array',
            'members.*' => 'exists:Users,User_id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $group->update($request->only([
            'Name',
            'Description',
            'Created_by',
        ]));

        if ($request->has('members')) {
            $group->members()->sync($request->input('members'));
        }

        return response()->json([
            'status' => 'success',
            'data' => $group,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $group = Group::findOrFail($id);
        $group->members()->detach();
        $group->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Group deleted successfully',
        ], 200);
    }
}
